/*
 * cSortedMsgQueue.cpp
 * 
 * Class implementatino for generic Fifo queue table.
 *
 * Authors:  Vicky & Ted
 */
#include "cSortedMsgQueue.h"
#include "cPbcastInfo.h"
#include "cPbcastNode.h"
#include "Protocol Stack/cProtocolStack.h"
#include "Protocol Stack/cMsgBuffer.h"
#include <assert.h>

#ifndef NULL
	#define NULL 0
#endif

/*
 * cSortedMsgQueue::cSortedMsgQueue()
 *
 * Purpose:	Create a new queue (circular, fifo).
 * IN:		size	-> The starting size of the queue.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is set up.
 * Throws:	cSortedMsgQueueException
 * Return:	-
 */
cSortedMsgQueue::cSortedMsgQueue(unsigned int size, DWORD msgTimeout, unsigned int capIncrement)
{
	mMsgTimeout		= msgTimeout;
	mSize			= size;
	mNumElements	= 0;
	mInsertPtr		= 0;
	mRemovePtr		= 0;
	mCapIncrement   = capIncrement;

	// Alloc queue space.
	mQueue = new MsgInfo[size];
	if(!mQueue)
	{
		throw cQueueException("<cSortedMsgQueue::cSortedMsgQueue>: Unable to alloc new queue.");
	}
	for(unsigned int i=0; i < size; i++)
	{
		mQueue[i].node = NULL;
		mQueue[i].msg  = NULL;
		mQueue[i].retransReqRemaining = 0;
	}
}

/*
 * cSortedMsgQueue::~cSortedMsgQueue()
 *
 * Purpose:	Destroy the queue.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is cleaned up.
 * Return:	-
 */
cSortedMsgQueue::~cSortedMsgQueue()
{
	mCS.Enter();
	if(mQueue) 
	{ 
		Clear();
		delete mQueue; 
	}
	mCS.Leave();
}

/*
 * cSortedMsgQueue::_Grow()
 *
 * Purpose:	Attempts to grow the queue by at least one, at most mCapIncrement
 * IN:		-
 * OUT:		-
 * Cond:	The queue is full!
 * PostCnd:	The queue is grown by at least one, at most mCapIncrement elements.
 * Return:	true if success, else false.
 */ 
bool cSortedMsgQueue::_Grow()
{
	// Attempt to compact the queue.
	Compact();

	if(mNumElements == (mSize-1))
	{
		// We need to expand the queue.
		MsgInfo* newQ = new MsgInfo[mSize+mCapIncrement];
		if(!newQ)
		{
			return false;
		}

		// The following may be faster using memcpy()...
		unsigned int copyLoc = 0;
		for(unsigned int i = mRemovePtr; i != mInsertPtr; i=(i+1) % mSize)
		{
			newQ[copyLoc] = mQueue[i];
			copyLoc++;
		}
		for(i = copyLoc; i < mSize+mCapIncrement; i++)
		{
			newQ[i].node = NULL;
			newQ[i].msg  = NULL;
			newQ[i].retransReqRemaining = 0;
		}
		mInsertPtr = copyLoc;
		mRemovePtr = 0;
		mSize = mSize + mCapIncrement;
		delete mQueue;
		mQueue = newQ;
	}
 	return true;
}

/*
 * cSortedMsgQueue::Insert()
 *
 * Purpose:	Inserts the msg in sorted order in the queue.
 * IN:		msg -> The message to insert.
 * OUT:		-
 * Cond:	The queue is not full.
 * PostCnd:	The free space in queue decreases by one.
 * Return:	true if success, else false.
 */ 
bool cSortedMsgQueue::Insert(MsgInfo msg)
{
	bool	retVal;
	int		searchResult;

	mCS.Enter();

	MsgInfo temp = msg; // used for searching

	// See if we need to replace.
	searchResult = _BinSearch(&temp, mRemovePtr, (mInsertPtr == 0) ? mSize-1 : mInsertPtr-1);
	if(searchResult != -1)
	{
		retVal = _Replace(&msg, searchResult);
		mCS.Leave();
		return retVal;
	}

	// Inserting brand new element!
	if(mNumElements == (mSize-1))
	{
		if(!_Grow())
		{
			mCS.Leave();
			return false;
		}
	}

	mQueue[mInsertPtr] = msg;
	if(mQueue[mInsertPtr].msg)
	{
		mQueue[mInsertPtr].msg->AddRef();
	}
	mQueue[mInsertPtr].node->AddRef();
	mQueue[mInsertPtr].timeInserted = cProtocolStack::GetTime();
	mNumElements++;

	// Sort it
	_InsertSort();

	mInsertPtr = (mInsertPtr +1) % mSize;
	retVal = true;

	mCS.Leave();
	return retVal;
}

/*
 * cSortedMsgQueue::_InsertSort()
 *
 * Purpose:	Keeps the sorted property of the queue, assuming new element is placed
 *			at mInsertPtr
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if a should be delivered before b, else false.
 */ 
bool cSortedMsgQueue::_InsertSort()
{
	unsigned int	current, prev;
	MsgInfo			temp;

	current = prev = mInsertPtr;
	while(current != mRemovePtr)
	{
		prev = (prev == 0) ? mSize-1 : prev-1;		// Wrap around?

		// Make sure that prev is not a hole!
		if(mQueue[prev].node)
		{
			if(!_Before(&mQueue[current], &mQueue[prev]))
			{
				return true;		// No need to swap anymore
			}
		}

		// Swap (into a hole if one exists)
		temp = mQueue[current];
		mQueue[current] = mQueue[prev];
		mQueue[prev] = temp;

		current = prev;
	}
	return true;
}

/*
 * cSortedMsgQueue::Replace()
 *
 * Purpose: Replaces the element at the given location with the new one.
 * IN:		info	-> The message info to replace the location with.
 * OUT:		-
 * Cond:	The old MsgInfo should be in the queue.
 * PostCnd:	The old MsgInfo is replaced w/ the new one.
 * Return:	true if success, else fail.
 */
bool cSortedMsgQueue::_Replace(MsgInfo* info, unsigned int location)
{
	if(info->msg)
	{
		info->msg->AddRef(); 
	}
	info->node->AddRef();
	if(mQueue[location].msg)
	{
		mQueue[location].msg->Release(); 
	}
	mQueue[location].node->Release();
	mQueue[location] = *info;
	mQueue[location].timeInserted = cProtocolStack::GetTime();	// WANT THIS???

	return true;
}

/*
 * cSortedMsgQueue::_Before()
 *
 * Purpose:	Returns true if a should be delivered before b
 * IN:		a, b	-> msgs to compare.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if a should be delivered before b, else false.
 */ 
bool cSortedMsgQueue::_Before(MsgInfo* a, MsgInfo* b)
{
	// Sort by node then by sequence number
	if(a->node->LessThan(b->node))
	{
		return true;
	}
	else if(a->node->Equals(b->node))
	{
		return cPbcastNode::SeqNumOlder(a->seqNum, b->seqNum);
	}
	else
	{
		return false;
	}
}

/*
 * cSortedMsgQueue::_Equal()
 *
 * Purpose:	Returns true if a refers to same message as b
 * IN:		a, b	-> msgs to compare.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	true if a is the same as b
 */ 
bool cSortedMsgQueue::_Equal(MsgInfo* a, MsgInfo* b)
{
	return (a->seqNum == b->seqNum) && (a->node->Equals(b->node));
}

/*
 * cSortedMsgQueue::Replace()
 *
 * Purpose: Searches for the element and replaces it with the new one.
 * IN:		searchMsg	-> The MsgInfo w/ parameters to search for set
 *						   These are: node, seqNum
 *						   And parameters to replace them with.  These
 *						   are: msg, retransReqRemaining
 * OUT:		-
 * Cond:	The old MsgInfo should be in the queue.
 * PostCnd:	The old MsgInfo is replaced w/ the new one.
 * Return:	true if success, else fail.
 */
bool cSortedMsgQueue::Replace(MsgInfo* searchMsg)
{
	mCS.Enter();

	MsgInfo temp = *searchMsg;
	int retVal;
	retVal = _BinSearch(&temp, mRemovePtr, (mInsertPtr == 0) ? mSize-1 : mInsertPtr-1);
	if(retVal != -1)
	{
		_Replace(searchMsg, retVal);
	}
	mCS.Leave();
	return (retVal != -1);
}

/*
 * cSortedMsgQueue::Find()
 *
 * Purpose:	Searches for the element and removes if necessary.
 * IN:		searchMsg	-> The MsgInfo w/ parameters to search for set
 *						   These are: node, seqNum
 * OUT:		searchMsg	-> Updated to hold the message if successful.
 * Cond:	The msg is in the queue.
 * PostCnd:	The msg is removed from the queue.
 * Return:	true if success, false otherwise.
 */
bool cSortedMsgQueue::Find(MsgInfo* searchMsg)
{
	int retVal;

	mCS.Enter();
	retVal = _BinSearch(searchMsg, mRemovePtr, (mInsertPtr==0) ? mSize-1 : mInsertPtr-1);
	mCS.Leave();

	return (retVal != -1);
}

/*
 * cSortedMsgQueue::_BinSearch()
 *
 * Purpose:	Performs binary search, looking for a match.
 * NOTE:	The input left/right can be wrap around values.
 * IN:		searchMsg	-> The MsgInfo w/ parameters to search for set
 *						   These are: node, seqNum
 *			left		-> The left bounds of the array (remove ptr)
 *			right		-> The right bounds of the array (insert ptr-1)
 * OUT:		searchMsg	-> Updated to hold the message if successful.
 * Cond:	The msg is in the queue.
 * PostCnd:	The searchMsg is updated with the message pointer.
 * Return:	-1 if not found, else the index in mQueue of the MsgInfo
 */
int cSortedMsgQueue::_BinSearch(MsgInfo* searchMsg, unsigned int left, unsigned int right)
{
	unsigned int pivot;

	while(1)
	{
		if(left == right)
		{
			if( _Equal(searchMsg, &mQueue[left]) )
			{
				*searchMsg = mQueue[left];
				return left;
			}
			else { return -1; }
		}
		else if(left < right)
		{
			pivot = (right-left) / 2;	// Just how much pivot advances from left, actually.
		}
		else // left > right
		{
			pivot = ((mSize - left) + right) / 2;	// Just how much pivot adv from left, actually.
		}
		pivot = (left + pivot) % mSize;	

		// If we are on a hole, keep moving to the right.
		while(mQueue[pivot].node == NULL)
		{
			pivot = (pivot + 1) % mSize;
			if(pivot == right)
			{
				return -1;	// Can't find it anywhere!
			}
		}

		// See where to search next
		if(_Before(searchMsg, &mQueue[pivot]))
		{
			// Search from left to pivot-1
			if(pivot == left)
			{
				return -1;	// No sense looping around again.
			}
			right = ((pivot==0) ? mSize-1 : pivot-1);
		}
		else if( _Equal(searchMsg, &mQueue[pivot]) )
		{
			// Found it!
			*searchMsg = mQueue[pivot];
			return pivot;
		}
		else
		{
			// Search from pivot+1 to right
			left = (pivot+1)%mSize;
		}
	}
}

/*
 * cSortedMsgQueue::Compact()
 *
 * Purpose:	Compacts the message queue (gets rid of holes and old timers)
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The message queue may be compacted from the rear onwards.
 * Return:	true if success, false otherwise.
 */
bool cSortedMsgQueue::Compact()
{
	mCS.Enter();

	DWORD	now = cProtocolStack::GetTime(); 
	unsigned int temp = mRemovePtr;
	unsigned int current;
	unsigned int prev;

	// Algorithm: Step through the queue from head to tail.  For each NULL or too old message,
	//			  bubble it back all the way to the head.  Then increase the head pointer.
	while(temp != mInsertPtr)
	{
		if( (mQueue[temp].node == NULL) || ((now - mQueue[temp].timeInserted) > mMsgTimeout) )
		{
			if(mQueue[temp].node != NULL)
			{
				// Timeout: know this because second cond of || was true!
				mQueue[temp].node->Release();	
				mQueue[temp].node = NULL;
				if(mQueue[temp].msg)
				{
					mQueue[temp].msg->Release();	
					mQueue[temp].msg  = NULL;
				}
			}
			current = temp;
			while(current != mRemovePtr)
			{
				prev = (current == 0) ? mSize-1 : current-1; // Wrap around?
				mQueue[current] = mQueue[prev];
				current = prev;
			}
			mQueue[current].node = NULL;	// Ensure that we have bubbled out any HOLES.
			mRemovePtr = (mRemovePtr + 1) % mSize;	 // Move the remove pointer forward.
			mNumElements--;
		}
		temp = (temp + 1) % mSize;
	}

	mCS.Leave();
	return true;
}

/*
 * cSortedMsgQueue::Clear()
 *
 * Purpose:	Clears out all of that node's messages.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is cleared of that node's messages.
 * Return:	true if success, false otherwise.
 */
bool cSortedMsgQueue::Clear(cPbcastNode* node)
{
	mCS.Enter();

	unsigned int temp = mRemovePtr;

	while(temp != mInsertPtr)
	{
		if(mQueue[temp].node)
		{
			if(mQueue[temp].node->Equals(node))
			{
				if(mQueue[temp].msg)
				{
					mQueue[temp].msg->Release();
					mQueue[temp].msg = NULL;
				}
				mQueue[temp].node->Release();
				mQueue[temp].node = NULL;
				// mNumElements--; ?????
			}
		}
		temp = (temp + 1) % mSize;
	}
	mCS.Leave();
	return true;
}

/*
 * cSortedMsgQueue::Clear()
 *
 * Purpose:	Clears out and resets the queue.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is reset and cleared.
 * Return:	true if success, false otherwise.
 */
bool cSortedMsgQueue::Clear()
{
	mCS.Enter();

	for(unsigned int i = 0; i < mSize; i++)
	{
		if(mQueue[i].node)
		{
			mQueue[i].node->Release();
			mQueue[i].node = NULL;
			if(mQueue[i].msg)
			{
				mQueue[i].msg->Release();
				mQueue[i].msg  = NULL;
			}
		}
	}
	mNumElements = 0;
	mInsertPtr   = 0;
	mRemovePtr   = 0;

	mCS.Leave();
	return true;
}
